/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/


package cnrg.itx.datax.devices;

import java.io.*;
import cnrg.itx.datax.*;

/**
 * Class to define a stream of data. Any valid output stream can be
 * represented by this class as a valid <code>Destination</code> object.
 */
public class StreamDestination implements Destination
{
	/**
	 * Attribute for storing the stream destination statistics.
	 */
	private Stats streamStats;
	
	/**
	 * Private attribute to store the output stream object
	 */
	private OutputStream outStream;
	
	/**
	 * Attribute for storing the number of bytes written to the stream.
	 */
	private int bytesRead = 0;
	
	/**
	 * Attribute for storing the device instance number. This is a static attribute and will be
	 * incremented in the constructor. This represents the total number of instances of
	 * the stream destinations.
	 */
	private static int instances = 0;
	
	/**
	 * Attribute for storing the device ID. This is the value of the instance at the moment.
	 */
	private int iDevice;
	
	/**
	 * Constructor to create a stream destination.
	 * @param OutputStream The output stream for this destination
	 * @exception DataException This is thrown in case of IO errors
	 */
	public StreamDestination(OutputStream outStream) throws DataException
	{
		// Increment the number of instances and assign the device id
		instances++;
		iDevice = instances;
		
		this.outStream = outStream;
		
		streamStats = new Stats();
	}
	
	/**
	 * Method to write data to the stream.
	 * @param b The array of bytes to write to the stream
	 * @exception DataException This is thrown when an IO error occurs
	 */
	public void write(byte [] b) throws DataException
	{
		try
		{
			outStream.write(b);
		}
		catch (IOException e)
		{
			throw new DataException(e.toString());
		}
		
		bytesRead += b.length;
		streamStats.addStat("<Stream Destination " + iDevice + "> Bytes read ", new Integer(bytesRead));
	}
	
	/**
	 * Method to close the stream
	 */
	public void close()
	{
		try
		{
			outStream.close();
		}
		catch(IOException E)
		{
			// Do nothing
		}
	}
	
	/**
	 * Method to mute the stream. This method is a no op as there is nothing
	 * to mute in an output stream.
	 * @param mute The mode of the mute
	 * @return boolean The mute state of the stream
	 */
	public boolean mute(boolean mute)
	{
		return true; 
	}
	
	/**
	 * Method to get the statistics of the output stream.
	 * @return Stats The statistics of the output stream
	 */
	public Stats getStatistics ()
	{
		return streamStats; 
	}
	
	/**
	 * Returns a collection of properties supported.
	 */ 
	public PropertiesCollection getProperties() throws DataException
	{
		return null;
	}

	/**
	 * Sets the given properties collection into the device
	 */
	public void setProperties(PropertiesCollection pc) throws DataException
	{
	}

	/**
	 * Interface to set the given properties collection into the device. WOrks under the 
	 * assumption that this is the properties collection of the peer.
	 */
	public void setPeerProperties(PropertiesCollection pc) throws DataException
	{
	}
}
